/**CHeaderFile*****************************************************************

  FileName    [spfdInt.h]

  PackageName [spfd]

  Synopsis    [Internal data structures for the spfd package.]

  Description [Internal data structures for the spfd package.]

  SeeAlso     [spfd.h]

  Author      [Balakrishna Kumthekar] 

  Copyright   [This file was created at the University of Colorado at Boulder.
  The University of Colorado at Boulder makes no warranty about the suitability
  of this software for any purpose.  It is presented on an AS IS basis.]

******************************************************************************/

#ifndef _SPFDINT
#define _SPFDINT

#include "part.h"
#include "ntm.h"
#include "cmd.h"
#include "spfd.h"
#include "truesim.h"
#include <string.h>

#ifdef USE_VPR
#include "vpr.h"
#endif

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/

#define SPFD_NETWORK_APPL_KEY "Spfd_NetworkApplKey"
/* Methods to sort nodes during logic optimization. */
#define RANDOM 0 /* Choose a network node randomly */
#define MAXSWCAP 1 /* Choose the node that has the maximum switched capacitance.
		      Switched capacitance is the product of the node's output
		      capacitance (approximated by the fanout count) and the
		      average switching acitivity (switching probability). */
#define MAXFANOUT 2 /* Choose the node with maximum fanout */
#define MINSWCAP 3 /* Choose the node with minimum switched capacitance */
#define MINFANOUT 4 /* Choose the node with minimum fanout */

/* Methods of optimizing the selected node */
#define REDUCE_FANIN_WIRES 0 /* Fanin wires of the selected node are selected
				one at a time to attempt to remove it. */
#define OPTIMIZE_MAX_NODE 1 /* Compute a suitable alternate implementation at
			       the selected node. */
#define OPTIMIZE_FANIN_NODES 2 /* Compute a suitable alternate implementation at
			       the fanin nodes of the selected node. */
#define REDUCE_FANOUT_WIRES 3 /* Attempt to remove fanout wires of the
				 selected node. */

/* Used during random value generation */
#define MODULUS1 2147483563

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/

/**Struct**********************************************************************

  Synopsis [Data structure used during simultaneous logic and
  placement optimization.]

  Description [Data structure used during simultaneous logic and
  placement optimization. This data structure stores information
  extracted from the data structures in VPR.]

  SeeAlso     [SpfdNodeData_t]

******************************************************************************/
typedef struct SpfdPlaceData_t {
  st_table *nodeToNetIndex; /* Table to store the index of net in nets[] (VPR)*/
  st_table *wireDelayTable; /* Table to store wire delays */
  st_table *nodeDelayTable; /* Table to store node delays */
  st_table *wireSeen; /* Wires processed. Used only in timing-based method */
} SpfdPlaceData_t;


/**Struct**********************************************************************

  Synopsis [The package application data structure.]

  Description [The package application data structure. This data
  structure is hooked to the network for easy access.]

  SeeAlso     [SpfdPlaceData_t SpfdNodeData_t]

******************************************************************************/
typedef struct SpfdApplData_t {
  bdd_manager *ddManager; /* BDD manager */
  bdd_node *currXCube; /* BDD of cube of primary inputs that are in the
			  transitive fanin of the network region under
			  optimization.   */
  st_table *nodeToData; /* Table of node and SpfdNodeData_t */
  st_table *currRegionNodes; /* internal and boundary nodes of the region.
				The region under consideration includes nodes
				(upto a user specified depth) in the
				transitive fanout of the selected node. */
  st_table *currBddReq; /* BDDs of the nodes in the current region */
  st_table *currInUseVars; /* BDD variable IDs currently in use. This is
			      necessary because we reuse previously allocated
			      BDD IDs. */
  st_table *currPiAltVars; /* Extra (auxillary variables) BDD variable IDs
			      for PIs in the transitive fanin of the region */
  st_table *currWireTable; /* Table to store wires that can be removed */
  st_table *currReplaceTable; /* Table to store wires that can be replaced */
  st_table *wiresRemoved; /* Table to store wires that have been removed */
  st_table *nodesRemoved; /* Table to store nodes that can be removed */
  SpfdPlaceData_t *placeData; /* Struct to store placement data */
} SpfdApplData_t;


/**Struct**********************************************************************

  Synopsis [Data structure to store additional information in a node.]

  Description [Data structure to store additional information in a
  node. The information stored is derived while computing SPFDs in the
  network.]

  SeeAlso     [SpfdPlaceData_t]

******************************************************************************/
typedef struct SpfdNodeData_t {
  bdd_node *spfd; /* Node SPFD */
  bdd_node *localAlt; /* An alternate function (implementation) chosen from
			 the node's SPFD */
  bdd_node *alternative; /* localAlt in terms of primary inputs */
  bdd_node **parameters; /* Binary parameters used during SPFD computation*/
  int numParams; /* size of parameters */
  array_t *faninOrder; /* Order of fanin nodes. The SPFDs for fanin nodes are
			  assigned according to this order. */
  int auxId; /* Two sets of variables are required to represent
                SPFDs. mddId of the node is one, and auxId is the
                other. */
  boolean locked; /* Set to TRUE after the node is optimized */
} SpfdNodeData_t;


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
extern int spfdCreatedPart; /* Whether network BDDs were created in the package */
extern int spfdVerbose; /* verbosity */
extern float spfdAlpha; /* spfdAlpha * load + (1-spfdAlpha) * topDepth, is used to order
                fanin nodes during SPFD computation */
extern boolean spfdPerfSim; /* Perform simulation */
extern boolean spfdNtkChanged; /* TRUE if network changes structurally or functionally */
extern boolean spfdReprogNoWire; /* If TRUE, no functional changes are performed
                         unless required by structural changes */
extern int spfdWiresAdded, spfdNumWiresRem; /* Number of wires added and removed */
extern int spfdSortNodes; /* Method to sort nodes */
extern int spfdMethod; /* Optimization method */

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

EXTERN boolean SpfdNodeReadLocked(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN int SpfdNodeReadAuxId(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeSetAuxId(SpfdApplData_t *applData, Ntk_Node_t *node, int auxId);
EXTERN bdd_node * SpfdNodeReadSpfd(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeDeleteSpfd(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeSetSpfd(SpfdApplData_t *applData, Ntk_Node_t *node, bdd_node *spfd);
EXTERN bdd_node ** SpfdNodeReadParameters(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeDeleteParameters(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeSetParameters(SpfdApplData_t *applData, Ntk_Node_t *node, bdd_node **parameters, int numParams);
EXTERN array_t * SpfdNodeReadFaninOrder(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeDeleteFaninOrder(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeSetFaninOrder(SpfdApplData_t *applData, Ntk_Node_t *node, array_t *faninOrder);
EXTERN bdd_node * SpfdNodeReadGlobalAlternative(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeDeleteGlobalAlternative(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeSetGlobalAlternative(SpfdApplData_t *applData, Ntk_Node_t *node, bdd_node *alternative);
EXTERN bdd_node * SpfdNodeReadLocalAlt(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeDeleteLocalAlt(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdNodeSetLocalAlt(SpfdApplData_t *applData, Ntk_Node_t *node, bdd_node *localAlt);
EXTERN void SpfdNodeSetLocked(SpfdApplData_t *applData, Ntk_Node_t *node, boolean locked);
EXTERN int SpfdNodeReadNumParams(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN int SpfdNodeReadNetIndex(SpfdApplData_t *applData, Ntk_Node_t *node);
EXTERN void SpfdApplFreeCallback(void *data);
EXTERN enum st_retval SpfdStBddFree(char *key, char *value, char *arg);
EXTERN enum st_retval SpfdStTableClean(char *key, char *value, char *arg);
EXTERN void SpfdReleaseAndCleanNodeData(SpfdApplData_t *applData, Ntk_Node_t *regNode, st_table *nodeTable);
EXTERN void SpfdReleaseAlternatePiIds(SpfdApplData_t *applData, Ntk_Node_t *regNode, st_table *nodeTable);
EXTERN void SpfdCleanUpAuxIds(SpfdApplData_t *applData);
EXTERN void SpfdRegionComputeSinglePairSpfd(Ntk_Network_t *network, SpfdApplData_t *applData, array_t *regionArray);
EXTERN bdd_node * SpfdNodeComputeOptParams(SpfdApplData_t *applData, Ntk_Node_t *regNode, bdd_node *result, bdd_node **parameters, int numIsfs);
EXTERN void SpfdNodeReduceSCCToSinglePair(SpfdApplData_t *applData, Ntk_Node_t *regNode, st_table *SCC);
EXTERN void SpfdComputeRequiredGlobalBdds(Ntk_Network_t *network, SpfdApplData_t *applData);
EXTERN int SpfdNetworkOptimize(Ntk_Network_t *network, char *simFile, int percent, int frequency, int regionDepth);
EXTERN void SpfdProcessFanoutWires(Ntk_Network_t *network, Ntk_Node_t *maxNode, int regionDepth, boolean replRem);
EXTERN Ntk_Node_t * SpfdCheckIfWireIsRedundantOrReplaceable(Ntk_Network_t *network, SpfdApplData_t *applData, Ntk_Node_t *from, Ntk_Node_t *to, array_t *replaceArray);
EXTERN Ntk_Node_t * SpfdTestIfNodeGlobalFuncSatisfiesWireSpfd(Ntk_Network_t *network, array_t *replaceArray, Ntk_Node_t *from, Ntk_Node_t *to, bdd_node *wireSpfd);
EXTERN void SpfdOptimizeFaninWires(Ntk_Network_t *network, Ntk_Node_t *maxNode, int regionDepth, boolean replRem);
EXTERN void SpfdOptimizeFanoutWires(Ntk_Network_t *network, Ntk_Node_t *maxNode, int regionDepth, boolean replRem);
EXTERN void SpfdOptimizeFaninNodes(Ntk_Network_t *network, Ntk_Node_t *maxNode, int regionDepth);
EXTERN void SpfdOptimizeWire(Ntk_Network_t *network, Ntk_Node_t *maxNode, Ntk_Node_t *ntkNode, int regionDepth, boolean replRem);
EXTERN void SpfdOptimizeNode(Ntk_Network_t *network, Ntk_Node_t *ntkNode, int regionDepth);
EXTERN void SpfdReprogramRegionNodes(Ntk_Network_t *network, SpfdApplData_t *applData, array_t *regionArray);
EXTERN void SpfdReprogramNode(Ntk_Network_t *network, SpfdApplData_t *applData, Ntk_Node_t *regNode);
EXTERN void SpfdNetworkRemoveWire(Ntk_Network_t *network, Ntk_Node_t *from, Ntk_Node_t *to);
EXTERN void SpfdNetworkAddWire(Ntk_Network_t *network, Ntk_Node_t *from, Ntk_Node_t *to);
EXTERN array_t * SpfdNodeComputeFanoutRegion(SpfdApplData_t *applData, Ntk_Node_t *startNode, int regionDepth);
EXTERN array_t * SpfdNodeComputeTFIUntilDepth(Ntk_Node_t *startNode, int regionDepth);
EXTERN void SpfdNodesInTFO(Ntk_Network_t *network, Ntk_Node_t *node, st_table *tfoNodes);
EXTERN bdd_node * SpfdNodeComputeSpfdFromOnAndOffSet(SpfdApplData_t *applData, Ntk_Node_t *regNode, bdd_node *bdd1, bdd_node *bdd0);
EXTERN bdd_node * SpfdNodeComputeSpfdFromFanouts(SpfdApplData_t *applData, Ntk_Node_t *from);
EXTERN st_table * SpfdNodeComputeSCCs(SpfdApplData_t *applData, Ntk_Node_t *regNode, bdd_node **tempVars);
EXTERN SpfdApplData_t * SpfdInitializeApplData(Ntk_Network_t *network);
EXTERN bdd_node ** SpfdComputeParameters(SpfdApplData_t *applData, st_table *SCC);
EXTERN bdd_node ** SpfdAllocateTemporaryVariables(bdd_manager *ddManager, st_table *inUseVars, int num);
EXTERN void SpfdAllocateOrReuseAuxVariables(Ntk_Network_t *network, SpfdApplData_t *applData);
EXTERN void SpfdOrderFaninOfRegionNodes(Ntk_Network_t *network, SpfdApplData_t *applData, int (*compareFunc)(const void *, const void *));
EXTERN int SpfdSwitchedCapAndDepthCompare(const void *obj1, const void *obj2);
EXTERN int SpfdFanoutCountAndDepthCompare(const void *obj1, const void *obj2);
EXTERN int SpfdDepthCompare(const void *obj1, const void *obj2);
EXTERN int SpfdDescendDepthCompare(const void *obj1, const void *obj2);
EXTERN void SpfdMddCreateVariables(mdd_manager *mgr, int numVarsToBeAdded, int level);
EXTERN bdd_node * SpfdAddEqual(bdd_manager *ddManager, bdd_node **f, bdd_node **g);
EXTERN int SpfdNetworkWriteBlifFile(Ntk_Network_t *network, char *fileName);
EXTERN bdd_node * SpfdComputeNodeArrayRelation(SpfdApplData_t *applData, st_table *currBddReq, array_t *nodeBdds, array_t *nodeArray, boolean useMddIds, int *piSwap);
EXTERN bdd_node * SpfdSwapPiAndAltPi(SpfdApplData_t *applData, bdd_node *fun);
EXTERN bdd_node * SpfdNodeReadLocalBdd(Ntk_Network_t *network, Ntk_Node_t *node);
EXTERN array_t * SpfdFetchInternalPatternArray(array_t *patternArray, int percent, double randomValue);
EXTERN Ntk_Node_t * SpfdFindNode(Ntk_Network_t *network, array_t *nodeArray);

/**AutomaticEnd***************************************************************/

#endif /* _SPFDINT */
