/**CHeaderFile*****************************************************************

  FileName    [tbl.h]

  PackageName [tbl]

  Synopsis    [Routines for manipulating a multi-valued relation representation.]

  Description [A Tbl_Table_t is a data structure that contains all the
  information found in the blif_mv tables (refer to blif_mv document), and
  represents a multivalued function or group of functions.
  This can be thought of as a table with inputs and outputs, and the
  entries in this table describe how the inputs are related to the outputs.

  The entries in a table may be of two types: they are either a list of ranges
  that contain values that an entry may take, or are set equal to some other
  entry in the table. Notice that blif_mv files also have the complement
  construct, which is absent in the table struct. There are functions for
  complementing, and canonicalizing a  list of ranges]

  SeeAlso     [tblInt.h, ntk.h]

  Author      [ Gitanjali M. Swamy]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

  Revision    [$Id: tbl.h,v 1.15 2009/04/11 02:01:29 fabio Exp $]

******************************************************************************/

#ifndef _TBL
#define _TBL
#include "mvf.h"
#include "mvfaig.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                         */
/*---------------------------------------------------------------------------*/
typedef enum {
  Tbl_EntryEqual_c,
  Tbl_EntryNormal_c,
  Tbl_EntryUniverse_c,
  Tbl_EntryUnassigned_c
} Tbl_EntryType_t;

/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/
typedef struct TblEntryStruct Tbl_Entry_t;
typedef struct TblTableStruct Tbl_Table_t;
typedef struct TblDataStruct Tbl_Data_t;
typedef struct TblRowStruct Tbl_Row_t;
typedef struct TblRangeStruct Tbl_Range_t;

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/

/**Macro***********************************************************************

  Synopsis     [ Generates all entries of  the table.  ]

  Description  [ This is an iterator that generates all entries
		 of the  table in smallest column first, smallest row first
		 and input first order. This must be
		 supplied with int colNum, rowNum,i (flag to indicate output
		 or input,  and Tbl_Entry_t * entry]

  SideEffects  [ ]

  SeeAlso      [ Tbl_TableForeachOutputEntry]

******************************************************************************/
#define Tbl_TableForEachEntry(                                               \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* int          */ rowNum  /* The row number            */,                  \
/* int          */ colNum  /* The column number         */,                  \
/* int          */  i       /* The input output flag     */,                 \
/* Tbl_Entry_t * */ entry   /* The table entry */                            \
)                                                                            \
for(rowNum=0;rowNum<Tbl_TableReadNumRows(table); rowNum++)\
 for(i=0; i<2; i++)\
 for(colNum=0; \
    ((colNum< Tbl_TableReadNumVars(table,i))\
     &&(entry=Tbl_TableReadEntry(table,rowNum,colNum,i))); colNum++)

/**Macro***********************************************************************

  Synopsis     [ Generates all input entries of  the table.  ]

  Description  [ This is an iterator that generates all input entries
		 of the  table in smallest column first, smallest row first
		 order. This must be supplied with int colNum, rowNum,
		 Tbl_Row_t *row, and Tbl_Entry_t * entry]

  SideEffects  [ ]

  SeeAlso      [ Tbl_TableForeachOutputEntry]

******************************************************************************/
#define Tbl_TableForEachInputEntry(                                          \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* int           */ rowNum  /* The row number            */,                 \
/* int           */ colNum  /* The column number         */,                 \
/* Tbl_Entry_t * */ entry   /* The table entry */                            \
)                                                                            \
 for(rowNum=0;rowNum<Tbl_TableReadNumRows(table); rowNum++)\
 for(colNum=0; \
    ((colNum< Tbl_TableReadNumVars(table,0))\
     &&(entry=Tbl_TableReadEntry(table,rowNum,colNum,0))); colNum++)

/**Macro***********************************************************************

  Synopsis     [ Generates all output entries of  the table.  ]

  Description  [ This is an iterator that generates all entries
		 of the  table in smallest column first, smallest row first.
		 This must be supplied with int colNum, rowNum, Tbl_Row_t *row, and
		 Tbl_Entry_t * entry]

  SideEffects  [ ]

  SeeAlso      [ Tbl_TableForEachInputEntry]

******************************************************************************/
#define Tbl_TableForEachOutputEntry(                                         \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* int           */ rowNum  /* The row number            */,                 \
/* int           */ colNum  /* The column number         */,                 \
/* Tbl_Entry_t * */ entry   /* The table entry */                            \
)                                                                            \
 for(rowNum=0;rowNum<Tbl_TableReadNumRows(table); rowNum++)\
 for(colNum=0; \
    ((colNum< Tbl_TableReadNumVars(table,1))\
     &&(entry=Tbl_TableReadEntry(table,rowNum,colNum,1))); colNum++)

/**Macro***********************************************************************

  Synopsis     [ Generates all default entries in the table ]

  Description  [ Given a table this iterates over all its default entries]

  SideEffects  [ ]

  SeeAlso      [ ]

******************************************************************************/
#define Tbl_TableForEachDefaultEntry(                                        \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* Tbl_Entry_t * */ entry   /* The table entry */,                           \
/* int           */ index   /* The default entry index */                    \
)                                                                            \
 arrayForEachItem( Tbl_Entry_t*,Tbl_TableReadDefaults(table),index,entry)

/**Macro***********************************************************************

  Synopsis     [Generates all input Var_Variable_t* associated with the table]

  Description  [Given a table, this function iterates over all its input
  Var_Variable_t*. This macro also provides the column number associated
  with the var, and must be supplied with int index, and Var_Variable_t* var.]

  SideEffects  []

  SeeAlso      []

******************************************************************************/
#define Tbl_TableForEachInputVar(                                            \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* int           */ index   /* The var index */,                             \
/* Var_Variable_t **/ var   /* The table var */                              \
)                                                                            \
 arrayForEachItem( Var_Variable_t*,Tbl_TableReadInputVars(table),index,var)

/**Macro***********************************************************************

  Synopsis     [Generates all output Var_Variable_t* associated with the table]

  Description  [Given a table, this function iterates over all its output
  Var_Variable_t*. This macro also provides the column number associated
  with the var, and must be supplied with int index, and Var_Variable_t* var.]

  SideEffects  []

  SeeAlso      []

******************************************************************************/
#define Tbl_TableForEachOutputVar(                                           \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* int           */ index   /* The var index */,                             \
/* Var_Variable_t **/ var   /* The table var */                              \
)                                                                            \
 arrayForEachItem( Var_Variable_t*,Tbl_TableReadOutputVars(table),index,var)


/**Macro***********************************************************************

  Synopsis     [ Generates all items  in the output of a row of a  table ]

  Description  [ Given a row number , this item iterates over all its output
  entries.It provides the colnum and the entry at each iteration]

  SideEffects  [ ]

  SeeAlso      []

******************************************************************************/
#define Tbl_RowForEachOutputEntry(                                           \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* int           */ rowNum  /* The row number            */,                 \
/* Tbl_Entry_t * */ entry   /* The table entry */,                           \
/* int           */ colNum  /* The column number         */                  \
)                                                                            \
 for(colNum=0; \
    ((colNum< Tbl_TableReadNumVars(table,1))\
     &&(entry=Tbl_TableReadEntry(table,rowNum,colNum,1))); colNum++)

/**Macro***********************************************************************

  Synopsis     [ Generates all items  in the input of a row of a  table ]

  Description  [ Given a row number, this item iterates over all its input
  entries. It provides the colnum and the entry at each iteration]

  SideEffects  [ ]

  SeeAlso      []

******************************************************************************/
#define Tbl_RowForEachInputEntry(                                            \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* int           */ rowNum  /* The row number            */,                 \
/* Tbl_Entry_t * */ entry   /* The table entry */,                           \
/* int           */ colNum  /* The column number         */                  \
)                                                                            \
 for(colNum=0; \
    ((colNum< Tbl_TableReadNumVars(table,0))\
     &&(entry=Tbl_TableReadEntry(table,rowNum,colNum,0))); colNum++)


/**Macro***********************************************************************

  Synopsis     [Iterates over all the values in a Tbl_Entry_t]

  Description  [Given a Tbl_Entry_t, this function iterates over all
  values in the entry. This function has to supplied an int value, an
  lsGen gen, and a Tbl_Range_t * range;]

  SideEffects []

  SeeAlso      []

******************************************************************************/
#define Tbl_EntryForEachValue(                                               \
/* Tbl_Entry_t * */ entry   /* The table entry to iterate on */,             \
/* int           */ value   /* The value at given iteration   */,            \
/* lsGen         */ gen     /* The generator         */,                     \
/* Tbl_Range_t * */ range   /* The range for each item  */                   \
)                                                                            \
 lsForEachItem(Tbl_EntryReadList(entry), gen, range)\
 for(value=Tbl_RangeBegin(range); value<= Tbl_RangeEnd(range); value++)

#include "var.h"

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

EXTERN MvfAig_Function_t * Tbl_TableBuildNonDetConstantMvfAig(Tbl_Table_t *table, int outIndex, int mAigId, mAig_Manager_t *manager);
EXTERN MvfAig_Function_t * Tbl_TableBuildMvfAigFromFanins(Tbl_Table_t * table, int outIndex, array_t * faninArray, mAig_Manager_t *manager);
EXTERN Tbl_Entry_t* Tbl_EntryAlloc(Tbl_EntryType_t type);
EXTERN void Tbl_EntryFree(Tbl_Entry_t * entry);
EXTERN Tbl_Entry_t * Tbl_EntryDup(Tbl_Entry_t *entry);
EXTERN void Tbl_EntrySetValue(Tbl_Entry_t * entry, int val1, int val2);
EXTERN Tbl_Entry_t* Tbl_EntryMerge(Tbl_Entry_t* entry1, Tbl_Entry_t* entry2);
EXTERN void Tbl_EntryComplement(Tbl_Entry_t * entry, int min, int max);
EXTERN void Tbl_EntrySetEqual(Tbl_Entry_t * entry, int varCol);
EXTERN boolean Tbl_EntryCheckRange(Tbl_Entry_t * entry, int val1, int val2);
EXTERN boolean Tbl_EntryIsEqual(Tbl_Entry_t *entry);
EXTERN Var_Variable_t* Tbl_EntryReadActualVar(Tbl_Table_t *table, Tbl_Entry_t *entry);
EXTERN Var_Variable_t* Tbl_EntryReadVar(Tbl_Table_t *table, Tbl_Entry_t *entry);
EXTERN int Tbl_EntryReadVarIndex(Tbl_Entry_t * entry);
EXTERN int Tbl_EntryReadNumValues(Tbl_Entry_t * entry);
EXTERN boolean Tbl_EntryTestEqualEntry(Tbl_Entry_t * entrya, Tbl_Entry_t * entryb);
EXTERN boolean Tbl_EntryTestIntersectEntry(Tbl_Entry_t * entrya, Tbl_Entry_t * entryb);
EXTERN lsList Tbl_EntryReadList(Tbl_Entry_t * entry);
EXTERN Tbl_EntryType_t Tbl_EntryReadType(Tbl_Entry_t *entry);
EXTERN int Tbl_RangeBegin(Tbl_Range_t *range);
EXTERN int Tbl_RangeEnd(Tbl_Range_t *range);
EXTERN Tbl_Table_t* Tbl_TableCollapse(Tbl_Table_t* table1, Tbl_Table_t* table2, int index);
EXTERN Tbl_Table_t * Tbl_TableInvertBinaryInputColumn(Tbl_Table_t *table, int index);
EXTERN void Tbl_Init(void);
EXTERN void Tbl_End(void);
EXTERN Tbl_Table_t* Tbl_TableAlloc(void);
EXTERN void Tbl_TableFree(Tbl_Table_t * table);
EXTERN Tbl_Table_t * Tbl_TableSoftDup(Tbl_Table_t * table);
EXTERN Tbl_Table_t * Tbl_TableHardDup(Tbl_Table_t * table);
EXTERN boolean Tbl_TableDefaultSetEntry(Tbl_Table_t *table, Tbl_Entry_t *entry, int index);
EXTERN Tbl_Entry_t* Tbl_TableDefaultReadEntry(Tbl_Table_t *table, int index);
EXTERN int Tbl_TableAddRow(Tbl_Table_t * table);
EXTERN int Tbl_TableAddColumn(Tbl_Table_t * table, Var_Variable_t * var, int flag);
EXTERN Tbl_Table_t * Tbl_TableRowDelete(Tbl_Table_t *originalTable, int rowNumToDelete, array_t *freeArray);
EXTERN Var_Variable_t* Tbl_TableReadIndexVar(Tbl_Table_t * table, int index, int flag);
EXTERN Tbl_Entry_t * Tbl_TableReadEntry(Tbl_Table_t * table, int rowNum, int colNum, int flag);
EXTERN boolean Tbl_TableSetEntry(Tbl_Table_t * table, Tbl_Entry_t * newEntry, int i, int j, int flag);
EXTERN boolean Tbl_TableSetEntryDc(Tbl_Table_t * table, int i, int j, int flag);
EXTERN void Tbl_TableAddEntryRange(Tbl_Table_t * table, int i, int j, int val1, int val2, int flag);
EXTERN void Tbl_TableComplementEntry(Tbl_Table_t * table, int i, int j, int flag);
EXTERN void Tbl_TableSetEquality(Tbl_Table_t * table, int i, int j, int flag, Var_Variable_t * var);
EXTERN boolean Tbl_TableEntryIsDc(Tbl_Table_t * table, int i, int j, int flag);
EXTERN boolean Tbl_TableTestIsConstant(Tbl_Table_t * table, int outputColumnId);
EXTERN boolean Tbl_TableTestIsNonDeterministicConstant(Tbl_Table_t * table, int outputColumnId);
EXTERN int Tbl_TableTestIsDeterministic(Tbl_Table_t * table);
EXTERN boolean Tbl_TableTestIsOutputSpaceComplete(Tbl_Table_t *table, mdd_manager *mddMgr);
EXTERN array_t * Tbl_TableComputeMvfAndInputDependenciesOfOutput(Tbl_Table_t *table, mdd_manager *mddMgr, int outIndex, int *value);
EXTERN Tbl_Table_t* Tbl_TableCreateTrueSupportTableForOutput(Tbl_Table_t *table, Mvf_Function_t *outMvf, mdd_manager *mddMgr, int offset, int outIndex, array_t *varMap);
EXTERN boolean Tbl_TableSwapRows(Tbl_Table_t * table, int i, int j);
EXTERN boolean Tbl_TableSwapColumns(Tbl_Table_t * table, int i, int j, int flag);
EXTERN void Tbl_TableCanonicalize(Tbl_Table_t * table);
EXTERN boolean Tbl_TableRowDominatesRow(Tbl_Table_t *table, int rowa, int rowb, array_t *rowValArray);
EXTERN boolean Tbl_TableColDominatesCol(Tbl_Table_t *table, int cola, int colb, int flag, Tbl_Row_t *colValArray);
EXTERN boolean Tbl_TablesAreIdentical(Tbl_Table_t *tablea, Tbl_Table_t *tableb, int a, int b);
EXTERN mdd_t * Tbl_TableRowToMdd(Tbl_Table_t * table, mdd_manager * manager, int i, array_t * svArray);
EXTERN Mvf_Function_t * Tbl_TableBuildNonDetConstantMvf(Tbl_Table_t * table, int outIndex, int mddId, mdd_manager * mddMgr);
EXTERN Mvf_Function_t * Tbl_TableBuildMvfFromFanins(Tbl_Table_t * table, int outIndex, array_t * faninArray, mdd_manager * mddMgr);
EXTERN int Tbl_TableReadConstValue(Tbl_Table_t * table, int outputColumnId);
EXTERN Tbl_Table_t * Tbl_MddToTable(mdd_t * mdd, mdd_manager * manager, st_table * idtoVar, array_t * inputids);
EXTERN boolean Tbl_TableSubstituteVar(Tbl_Table_t * table, Var_Variable_t * oldVar, Var_Variable_t * newVar);
EXTERN void Tbl_TableSetVar(Tbl_Table_t * table, int i, Var_Variable_t * sv, int flag);
EXTERN array_t * Tbl_TableSplit(Tbl_Table_t * table);
EXTERN int Tbl_TableReadNumInputs(Tbl_Table_t * table);
EXTERN int Tbl_TableReadNumOutputs(Tbl_Table_t * table);
EXTERN int Tbl_TableReadNumVars(Tbl_Table_t *table, int flag);
EXTERN int Tbl_TableReadNumRows(Tbl_Table_t * table);
EXTERN int Tbl_TableReadVarIndex(Tbl_Table_t * table, Var_Variable_t * var, int flag);
EXTERN void Tbl_TablePrintStats(Tbl_Table_t *table, FILE *fp);
EXTERN void Tbl_TableWriteBlifMvToFileSpecial(Tbl_Table_t *table, int flag, FILE *fp);
EXTERN void Tbl_TableWriteBlifMvToFile(Tbl_Table_t *table, int flag, FILE *fp);
EXTERN void Tbl_TableWriteSmvToFile(Tbl_Table_t *table, int flag, FILE *fp);
EXTERN void Tbl_TableWriteBlifToFile(Tbl_Table_t *table, FILE *fp);
EXTERN array_t* Tbl_TableReadDefaults(Tbl_Table_t * table);
EXTERN array_t* Tbl_TableReadInputVars(Tbl_Table_t * table);
EXTERN array_t* Tbl_TableReadOutputVars(Tbl_Table_t * table);
EXTERN boolean Tbl_RowInputIntersect(Tbl_Table_t * table, int a, int b);
EXTERN boolean Tbl_RowOutputIntersect(Tbl_Table_t * table, int a, int b);
EXTERN boolean Tbl_TableIsIdentity(Tbl_Table_t *table);
EXTERN boolean Tbl_TableIsInverter(Tbl_Table_t *table);

/**AutomaticEnd***************************************************************/

#endif /*_TBL */
