/**CHeaderFile*****************************************************************

  FileName    [tblInt.h]

  PackageName [tbl]

  Synopsis    [Include the internals of the table package]

  Description [Describes the internal structs in the table package. The
  Tbl_Table_t data structure consists of an array of inputs, and array of
  outputs and a data struct (Tbl_Data_t). This data struct is shared between
  tables that are duplicated (using soft dup). The data struct contains a
  reference count to indicate the number of tables that share it. If a
  free is called, the data struct reference count is decremented and only
  if this drops to 0 is the struct actually freed. The Tbl_Data_t also
  contains an array of default values, and an array of Tbl_Row_t's to
  represent the actual data. The Tbl_row_t consists of two arrays (inputs and
  outputs), which contain Tbl_Entry_t's (representing entries in a table.]

  SeeAlso     [tbl.h]

  Author      [Gitanjali M. Swamy]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

  Revision    [$Id: tblInt.h,v 1.11 2009/04/11 02:01:29 fabio Exp $]

******************************************************************************/

#ifndef _TBLINT
#define _TBLINT

#include "tbl.h"
#include "cmd.h"
#include "io.h"
#include <string.h>

/*---------------------------------------------------------------------------*/
/* Stucture declarations                                                     */
/*---------------------------------------------------------------------------*/

/**Struct*************************************************************

  Synopsis	[ This is basic blif_mv table data structure ]

  Description	[ This datastructure consists of struct with 3 fields; an
  array of inputs, an array of ouputs and a Tbl_Data_t for the actual
  table data.]

  SeeAlso	[ TblEntryStruct TblRowStruct TblDataStruct]
**********************************************************************/
struct TblTableStruct {
    array_t * inputNames; /* An array of the Var_Variable_ts corresponding to
                               the inputs to the table */
    array_t * outputNames; /* An array of Var_Variable_ts corresponding to
                                the outputs of the table */
    Tbl_Data_t * data; /* A Tbl_Data_t that represents actual data in the table  */ 
};

/**Struct*************************************************************

  Synopsis	[ This is the struct used to represent each row in the table]

  Description	[ There are 2 fields in this struct; a row array and a column
                  array.]

  SeeAlso	[ TblStruct TblEntryStruct ]
  
**********************************************************************/

struct TblRowStruct {
    array_t * inputs; /* An array of inputs */
    array_t * outputs; /* An array of outputs */
};

/**Struct**********************************************************************

  Synopsis    [Struct used to represent table data]

  Description [This struct has three fields; a reference count that is used
  for memory management, an array of default values, and an array of Tbl_Row_t's
  that represents the actual table data. Each Tbl_Row_t within this data array
  has an input part and an output part; each of these is an array of Tbl_Entry_t's.]

  SeeAlso     [ TblEntryStruct TblTableStruct TblRowStruct]

******************************************************************************/

struct TblDataStruct {
    int refCount; /* a reference count to indicate how many duplicates exist
                   */
    array_t *defaults; /* An array with the defaults */  
    array_t *actualData;
};


/**Struct*************************************************************

  Synopsis	[ This struct represents a range]

  Description	[ This struct represents a range and has 2 fields;
                  the beginning and the end. The data within a Tbl_Entry_t
                  is represented as a linked list of such ranges (if the entry
                  is of type Tbl_EntryNormal_c.]

  SeeAlso	[ TblRangeAlloc TblRangeFree ]
  
**********************************************************************/

struct TblRangeStruct {
    int begin; /* The beginning of the range */
    int end; /* The end of the range */
};
    
/**Struct*************************************************************

  Synopsis	[ This is the struct used to represent each entry in the table ]

  Description	[ This struct has 4 fields; an enumerated  type, an integer to
  indicate if it is an input or output, an integer for the  column number, and
  a union with either a list of ranges or a var (this depends on the type of
  the struct.]

  SeeAlso	[ TblRowStruct TblTableStruct TblDataStruct]
  
**********************************************************************/

struct TblEntryStruct {
    Tbl_EntryType_t type; /* Type of the table entry. This may be one of 2 types;
                  either equal  or normal*/
    int ioType; /* to indicate whether its input or output */
    int varColNum; /* corresponding Var_Variable_t column number */
    union{
        int var; /* This indicates the related column  if the entry is
                           of the type equal */
        lsList  listOfRanges; /* This indicates the contents of the table
                                  entry if its type is normal */
    }EntryData;
};


/**Macro***********************************************************************

  Synopsis     [return refcount associated with table]

  Description  [This functions returns the number of tables that share data
  with the given table. This count is used for memory management.]

  SideEffects  []

  SeeAlso      []

******************************************************************************/
#define TblTableReadRefCount(                                                \
/* Tbl_Table_t * */ table   /* The table to iterate over */                  \
)                                                                            \
table->data->refCount

/**Macro***********************************************************************

  Synopsis    [Return the data array associated with a table.]

  Description []

  SideEffects [This array is NOT to be modified. The user is encouraged not
  to use this function. It is exported so as to make the macros for iteration over
  items in the table possible. The user should use these macros for accessing data.
  The macros that can be used are mentioned in the SeeAlso list]

  SeeAlso     [Tbl_TableForEachEntry Tbl_TableForEachInputEntry Tbl_TableForEachOutputEntry
  Tbl_TableForEachInputVar Tbl_TableForEachOutputVar Tbl_TableForEachDefault]

******************************************************************************/
#define TblTableReadData(                                                    \
/* Tbl_Table_t * */ table   /* The table to iterate over */                  \
)                                                                            \
table->data->actualData

/**Macro***********************************************************************

  Synopsis    [Return the defaults array associated with a table.]

  Description []

  SideEffects [This array is NOT to be modified. The user is encouraged not
  to use this function. It is exported so as to make the macros for iteration over
  items in the table possible. The user should use these macros for accessing data.
  The macros that can be used are mentioned in the SeeAlso list]

  SeeAlso     [Tbl_TableForEachEntry Tbl_TableForEachInputEntry Tbl_TableForEachOutputEntry
  Tbl_TableForEachInputVar Tbl_TableForEachOutputVar Tbl_TableForEachDefault]

******************************************************************************/
#define TblTableReadDefaults(                                                \
/* Tbl_Table_t * */ table   /* The table to iterate over */                  \
)                                                                            \
 table->data->defaults


/**Macro***********************************************************************

  Synopsis     [ Generates all rows on the table ]

  Description  [ Given a table this iterates over all its rows in smallest colum
n
                 first order.]

  SideEffects  [ ]

  SeeAlso      [ ]

******************************************************************************/
#define TblTableForEachRow(                                                  \
/* Tbl_Table_t * */ table   /* The table to iterate over */,                 \
/* Tbl_Row_t *  */  row    /* The current row */,                            \
/* int          */ rowNum  /* The row number  */                             \
)                                                                            \
 for(rowNum=0; (rowNum <array_n(TblTableReadData(table)))\
                   &&(row = array_fetch(Tbl_Row_t*,(TblTableReadData(table)),rowNum));rowNum++)

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Function prototypes                                                       */
/*---------------------------------------------------------------------------*/

EXTERN mAigEdge_t TblEntryNormalConstructAig(mAig_Manager_t *manager, Tbl_Entry_t *entry, array_t *mAigArray);
EXTERN mAigEdge_t TblEntryEqualConstructMAig(mAig_Manager_t * manager, Tbl_Entry_t * entry, MvfAig_Function_t * mAigArray, MvfAig_Function_t * mEigArray);
EXTERN mdd_t * TblEntryNormalConstructMdd(mdd_manager *manager, Tbl_Entry_t * entry, array_t * mddArray);
EXTERN mdd_t * TblEntryEqualConstructMdd(mdd_manager *manager, Tbl_Entry_t * entry, Mvf_Function_t * mddArray, Mvf_Function_t * mddEArray);
EXTERN void TblEntryWriteBlifMv(Tbl_Table_t *table, Tbl_Entry_t *entry, FILE *fp);
EXTERN void TblEntryWriteSmv(Tbl_Table_t *table, Tbl_Entry_t *entry, boolean varnameflag, FILE *fp);
EXTERN void TblEntryWriteBlif(Tbl_Table_t *table, Tbl_Entry_t *entry, FILE *fp);
EXTERN void TblRangeFree(Tbl_Range_t * range);
EXTERN boolean TblTableDeleteLastRow(Tbl_Table_t* table);
EXTERN void TblTableSetRow(Tbl_Table_t *table, Tbl_Row_t *row, int i);
EXTERN array_t* TblRowReadOutputs(Tbl_Row_t * row);
EXTERN array_t* TblRowReadInputs(Tbl_Row_t * row);
EXTERN Tbl_Row_t* TblTableReadRow(Tbl_Table_t *table, int rowNum);
EXTERN Tbl_Row_t* TblRowAlloc(void);
EXTERN Tbl_Row_t* TblRowDup(Tbl_Row_t * row);
EXTERN void TblRowSetEntry(Tbl_Row_t *row, Tbl_Entry_t * entry, int i, int flag);
EXTERN void TblRowFree(Tbl_Row_t *row);
EXTERN Tbl_Entry_t* TblRowReadEntry(Tbl_Row_t *row, int i, int flag);

/**AutomaticEnd***************************************************************/

#endif /* _TBLINT */
