/**CFile***********************************************************************

  FileName    [truesimSim.c]

  PackageName [truesim]

  Synopsis    [Top-level routine to perform simulation.]

  Author      [Balakrishna Kumthekar]

  Copyright [This file was created at the University of Colorado at Boulder.
  The University of Colorado at Boulder makes no warranty about the suitability
  of this software for any purpose.  It is presented on an AS IS basis.]

******************************************************************************/

#include "truesimInt.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

/* Global variable from truesimCmd.c. */
extern int truesimRptHeader;
extern int truesimVerbose;

/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/


/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/
/**Function********************************************************************

  Synopsis           [Entry routine to perform network simulation.]

  Description        [Entry routine to perform network simulation.
  simFile provides bit-vectors for simulation. In the absence of
  simFile, probFile provides the primary input probabilites using with
  random bit-vector patterns are generated. delayFile has the
  <delay,load> pair for each node in the circuit. If simFile is not
  specified and random vectors are generated, those vectors are dumped
  to dumpFile. If genVectors is 0, no vectors are generated.
  
  Some of the arguments are redundant, but they are kept here for
  historical reasons.]

  SideEffects        [required]

******************************************************************************/
int
TruesimSimulateNetwork(
  Ntk_Network_t *network,
  char *simFile,
  char *probFile,
  char *delayFile,
  char *dumpFile,
  boolean trueDelay,
  int genVectors,
  int N)
{
  array_t *inputArray,*patternArray;
  int i,status;
  char *str;
  
  /* inputArray stores the names of primary input nodes and patternArray stores
     the simulation vectors as char strings. One could use bit packing, but
     this is fast and simple. */
  inputArray = array_alloc(Ntk_Node_t *,0);
  patternArray = array_alloc(char *,0);

  if (genVectors == 0) { /* Read simulation file */
    Truesim_ReadSimulationVectors(network,simFile,inputArray,patternArray);
  } else { /* Generate random vectors */
    array_t *probArray;
    int numInputs;
    
    probArray = array_alloc(float,0);
    Truesim_ReadInputProbabilities(network,probFile,inputArray,probArray);

    if(array_n(probArray) == 0) {
      array_free(inputArray);
      array_free(patternArray);
      array_free(probArray);
      return 0;
    }
    numInputs = array_n(inputArray);
    Truesim_GenerateRandomVectors(network,probArray,patternArray,
				  numInputs,N);
    array_free(probArray);

    /* Output pattern vectors */
    if (dumpFile) {
      Truesim_DumpSimulationVectors(network,inputArray,patternArray,dumpFile);
    }
  }

  /* Initialize the simulation data structures */
  Truesim_InitializeSimulation(network,delayFile,trueDelay,truesimRptHeader,
			       truesimVerbose,NIL(st_table));

  /* Now perform true delay simulation or zero delay simulation
     according to 'trueDelay' */  
  if (trueDelay) { /* Perform event driven simulation */
    status = Truesim_RealDelayPatternSimulate(network,inputArray,patternArray);
  } else { /* Perform zero-delay simulation */
    status = Truesim_ZeroDelayPatternSimulate(network,inputArray,patternArray);
  }

  /* End the simulation */
  Truesim_QuitSimulation(network);

  /* Free the space used to store the input nodes */
  array_free(inputArray);
  /* Free the pattern vectors */
  arrayForEachItem(char *, patternArray,i,str) {
    FREE(str);
  }
  array_free(patternArray);

  return status;

} /* End of TruesimSimulateNetwork */

/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

