/**CFile***********************************************************************

  FileName    [vmMain.c]

  PackageName [vm]

  Synopsis    [Main VIS routine. Parses command line at invocation of VIS.]

  Author      [Originated from SIS]

  Copyright   [Copyright (c) 1994-1996 The Regents of the Univ. of California.
  All rights reserved.

  Permission is hereby granted, without written agreement and without license
  or royalty fees, to use, copy, modify, and distribute this software and its
  documentation for any purpose, provided that the above copyright notice and
  the following two paragraphs appear in all copies of this software.

  IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
  DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
  OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
  CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

  THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
  FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN
  "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO PROVIDE
  MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.]

******************************************************************************/

#include "vmInt.h"

static char rcsid[] UNUSED = "$Id: vmMain.c,v 1.6 2004/07/16 00:09:59 wangc Exp $";

/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/
FILE *vis_stderr;
FILE *vis_stdout;
FILE *vis_historyFile;
FILE *vis_stdpipe;
array_t *vm_commandHistoryArray;
char *vm_programName;


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static int VisrcSource(Hrc_Manager_t ** hmgr);
static void UsagePrint(char * program);
static int TypeCheck(char * s);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis    [required]

  Description [optional]

  SideEffects [Sets vm_programName to the name of the excutable.]

  SeeAlso     [optional]

******************************************************************************/
int
main(
  int  argc,
  char ** argv)
{
  int c,
    status,
    batch,          /* 0 iff we have an interactive run (-c or -f) */
    initial_source, /* 1 iff we have to read .visrc                */
    initial_read,
    final_write;
  int quit_flag;
  char readcmd[20],
    writecmd[20];
  char *dummy,
    *cmdline,
    *cmdline1,
    *infile,        /* where to read input */
    *outfile;       /* where to write output */
  Hrc_Manager_t *globalHmgr;
  
  vm_programName = argv[0];
  quit_flag = -1;     /* Quick quit */

  VmInit();
  globalHmgr = Hrc_ManagerAlloc();

  cmdline = util_strsav("");
  (void) strcpy(readcmd, "read_blif_mv");
  (void) strcpy(writecmd, "write_blif_mv");
  infile = "-";
  outfile = "-";
  vm_commandHistoryArray = array_alloc(char *, 0);
  initial_source = 1;
  initial_read = 1;
  final_write = 1;
  batch = 0;
  util_getopt_reset();
  while ((c = util_getopt(argc, argv, "c:hf:o:st:T:x")) != EOF) {
    switch(c) {
	case 'c':
          FREE(cmdline);
          cmdline = util_strsav(util_optarg);
          batch = 1;
          break;

	case 'f':
          FREE(cmdline);
          cmdline = ALLOC(char, strlen(util_optarg) + 20);
          (void) sprintf(cmdline, "source %s", util_optarg);
          batch = 1;
          break;
          
	case 'h':
            UsagePrint(argv[0]);
            break;
             
	case 'o':
          outfile = util_optarg;
          break;

	case 's':
          initial_source = 0;
          break;

	case 't':
          if (TypeCheck(util_optarg)) {
            if (strcmp(util_optarg, "none") == 0) {
              initial_read = 0;
            }
            else {
              (void) sprintf(readcmd, "read_%s", util_optarg);
            }
          }
          else {
            UsagePrint(argv[0]);
          }
          batch = 1;
          break;

	case 'T':
          if (TypeCheck(util_optarg)) {
            if (strcmp(util_optarg, "none") == 0) {
              final_write = 0;
            }
            else {
              (void) sprintf(writecmd, "write_%s", util_optarg);
            }
          }
          else {
            UsagePrint(argv[0]);
          }
          batch = 1;
          break;

	case 'x':
          final_write = 0;
          initial_read = 0;
          batch = 1;
          break;

	default:
          UsagePrint(argv[0]);
    }
  }

  if (! batch) {
    /* interactive use ... */
    if (argc - util_optind != 0) {
      (void) fprintf(vis_stderr, "warning -- trailing arguments ignored\n");
    }

    (void) fprintf(vis_stdout, "%s\n", Vm_VisReadVersion());
    if (initial_source) {
      (void) VisrcSource(&globalHmgr);
    }
    while ((quit_flag = Cmd_CommandExecute(&globalHmgr, "source -ip -")) >= 0)
      ;
    status = 0;

  } else {

    if (argc - util_optind == 0) {
      infile = "-";
    }
    else if (argc - util_optind == 1) {
      infile = argv[util_optind];
    }
    else {
      UsagePrint(argv[0]);
    }

    if (initial_source) {
      (void) VisrcSource(&globalHmgr);
    }

    status = 0;
    if (initial_read) {
      cmdline1 = ALLOC(char, strlen(infile) + 20);
      (void) sprintf(cmdline1, "%s %s", readcmd, infile);
      status = Cmd_CommandExecute(&globalHmgr, cmdline1);
      FREE(cmdline1);
    }

    if (status == 0) {
      /* cmd line contains `source <file>' */
      status = Cmd_CommandExecute(&globalHmgr, cmdline);
      if ((status == 0 || status == -1) && final_write) {
        cmdline1 = ALLOC(char, strlen(outfile) + 20);
        (void) sprintf(cmdline1, "%s %s", writecmd, outfile);
        status = Cmd_CommandExecute(&globalHmgr, cmdline1);
        FREE(cmdline1);
      }
    }
    
    /* FIX: RB, allows a clean quit from a script */
    quit_flag = status;
    

  }

  FREE(cmdline);
  for (c = array_n(vm_commandHistoryArray); c-- > 0; ){
    dummy = array_fetch(char *, vm_commandHistoryArray, c);
    FREE(dummy);
  }

  array_free(vm_commandHistoryArray);
  /* Value of "quit_flag" is determined by the "quit" command */
  if (quit_flag == -1 || quit_flag == -2) {
    status = 0;
  }
  if (quit_flag == -2) {
    Hrc_ManagerFree(globalHmgr);
    VmEnd();
  }
  return(status);
}


/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/


/**Function********************************************************************

  Synopsis    [Sources the .visrc file.]

  Description [Sources the .visrc file.  Always sources the .visrc from
  library.  Then source the .visrc from the home directory.  If there is none
  in the home directory, then execute the one in the current directory if one
  is present.  Returns 1 if scripts were successfully executed, else return 0.]

  SideEffects []

  SeeAlso     [optional]

******************************************************************************/
static int
VisrcSource(
  Hrc_Manager_t ** hmgr)
{
  char *commandLine;
  char *libraryName;
  char *homeFile;
  struct stat home;
  struct stat cur;
  int s1;
  int s2;			/* flags for checking the stat() call */
  int status0;
  int status1 = TRUE;
  int status2 = TRUE;

  /*
   * First execute the standard .visrc.
   */
  libraryName = Vm_VisObtainLibrary();
  commandLine = ALLOC(char, strlen(libraryName) + 25);
  (void) sprintf(commandLine, "source -s %s/master.visrc", libraryName);
  FREE(libraryName);
  status0 = Cmd_CommandExecute(hmgr, commandLine);
  FREE(commandLine);

  /*
   * Look in home directory and current directory for .visrc.
   */
  homeFile = util_tilde_expand("~/.visrc");
  s1 = stat(homeFile, &home);
  FREE(homeFile);
  s2 = stat(".visrc", &cur);

  /*
   * If .visrc is present in both the home and current directories, then read
   * it from the home directory.  Otherwise, read it from wherever it's
   * located.
   */
  if ((s1 == 0) && (s2 == 0) && (home.st_ino == cur.st_ino)){
    /* ~/.visrc == .visrc : Source the file only once */
    status1 = Cmd_CommandExecute(hmgr, "source -s ~/.visrc");
  }
  else {
    if (s1 == 0) {
      status1 = Cmd_CommandExecute(hmgr, "source -s ~/.visrc");
    }
    if (s2 == 0) {
      status2 = Cmd_CommandExecute(hmgr, "source -s .visrc");
    }
  }

  return (status0 && status1 && status2);
}


/**Function********************************************************************

  Synopsis    [Prints the usage of the VIS shell interface.]

  SideEffects []

  SeeAlso     [optional]

******************************************************************************/
static void
UsagePrint(
  char * program)
{
  char *libraryName;
  
  (void) fprintf(vis_stderr,
                 "%s\n", Vm_VisReadVersion());
  (void) fprintf(vis_stderr,
                 "usage: %s [-c cmd] [-f script] [-h] [-o file] [-s] [-t type] [-T type] [-x] [file]\n", 
                 program);
  (void) fprintf(vis_stderr,
                 "    -c cmd\texecute VIS commands `cmd'\n");
  (void) fprintf(vis_stderr,
                 "    -f file\texecute VIS commands from a file\n");
  (void) fprintf(vis_stderr,
                 "    -h\t\tprint the command usage\n");
  (void) fprintf(vis_stderr,
                 "    -o file\tspecify output filename (default is -)\n");
  (void) fprintf(vis_stderr,
                 "    -s\t\tdo not read any initialization file\n");
  libraryName = Vm_VisObtainLibrary();
  (void) fprintf(vis_stderr,
                 "      \t\t(%s/master.visrc, ~/.visrc)\n", libraryName);
  FREE(libraryName);
  (void) fprintf(vis_stderr,
                 "    -t type\tspecify input type (blif_mv (default), blif, or none)\n");
  (void) fprintf(vis_stderr,
                 "    -T type\tspecify output type (blif_mv (default), blif, smv, or none)\n");
  (void) fprintf(vis_stderr,
                 "    -x\t\tequivalent to '-t none -T none'\n");

  exit(2);
}


/**Function********************************************************************

  Synopsis    [Returns 1 if s is a file type recognized by VIS, else returns 0.]

  Description [Returns 1 if s is a file type recognized by VIS, else returns
  0. Recognized types are "blif", "blif_mv", and "none".]

  SideEffects []

******************************************************************************/
static int
TypeCheck(
  char * s)
{
  if (strcmp(s, "blif") == 0) {
    return 1;
  }
  else if (strcmp(s, "blif_mv") == 0) {
    return 1;
  }
  else if (strcmp(s, "smv") == 0) {
    return 1;
  }
  else if (strcmp(s, "none") == 0) {
    return 1;
  }
  else {
    (void) fprintf(vis_stderr, "unknown type %s\n", s);
    return 0;
  }
}
