/**CFile***********************************************************************

  FileName    [spfdClean.c]

  PackageName [spfd]

  Synopsis [Routines to free memory when not needed.]

  Description [Routines to free memory when not needed.]

  SeeAlso     [spfdAPI.c spfdUtil.c]

  Author      [Balakrishna Kumthekar]

  Copyright [This file was created at the University of Colorado at Boulder.
  The University of Colorado at Boulder makes no warranty about the suitability
  of this software for any purpose.  It is presented on an AS IS basis.]

******************************************************************************/

#include "spfdInt.h"

/*---------------------------------------------------------------------------*/
/* Constant declarations                                                     */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Type declarations                                                         */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Structure declarations                                                    */
/*---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*/
/* Variable declarations                                                     */
/*---------------------------------------------------------------------------*/

extern int sfpdVerbose;
extern float alpha;

/*---------------------------------------------------------------------------*/
/* Macro declarations                                                        */
/*---------------------------------------------------------------------------*/


/**AutomaticStart*************************************************************/

/*---------------------------------------------------------------------------*/
/* Static function prototypes                                                */
/*---------------------------------------------------------------------------*/

static enum st_retval NodeDataFree(char *key, char *value, char *arg);

/**AutomaticEnd***************************************************************/


/*---------------------------------------------------------------------------*/
/* Definition of exported functions                                          */
/*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/* Definition of internal functions                                          */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis [Routine to free the application data structure attached by
  this package to network. Certain fields of the SpfdApplData_t are
  freed earlier.]

  SideEffects [None]

******************************************************************************/
void
SpfdApplFreeCallback(void *data)
{
  SpfdApplData_t *applData = (SpfdApplData_t *)data;
  st_table *nodeToData = applData->nodeToData;
  bdd_manager *mgr = applData->ddManager;
  
  if (applData->currWireTable)
    st_free_table(applData->currWireTable);
  
  if (applData->currReplaceTable)
    st_free_table(applData->currReplaceTable);
  
  if (applData->wiresRemoved)
    st_free_table(applData->wiresRemoved);

  if (applData->nodesRemoved) {
    st_free_table(applData->nodesRemoved);
  }
  
  if (nodeToData) {
    st_foreach(nodeToData,NodeDataFree,(char *)mgr);
    st_free_table(nodeToData);
  }

  /* The data private to placeData is assumed to be deleted elsewhere */
  if (applData->placeData)
    FREE(applData->placeData);

  /* The rest of the fields are deleted once elsewhere. */
  
  FREE(applData);
  
  return;

} /* End of SpfdApplFreeCallback */


/**Function********************************************************************

  Synopsis           [Free BDDs stored in a st_table.]

  SideEffects        [None]

******************************************************************************/
enum st_retval
SpfdStBddFree(
  char *key,
  char *value,
  char *arg)
{
  bdd_recursive_deref((bdd_manager *)arg,(bdd_node *)value);
  
  return (ST_CONTINUE);

} /* End of SpfdStBddFree */


/**Function********************************************************************

  Synopsis [This function frees the table stored as 'value' in an st_table and
  returns ST_DELETE so that memory allocated for 'key' is also freed. In the
  end the table is 'cleaned' but not 'freed.]

  SideEffects        [None]

******************************************************************************/
enum st_retval
SpfdStTableClean(
  char *key,
  char *value,
  char *arg)
{
  st_free_table((st_table *)value);

  return (ST_DELETE);
} /* End of SpfdStTableClean */


/**Function********************************************************************

  Synopsis [Release those ids that will no longer be necessary, also
  delete global and local BDDs. After an alternate implementation at
  'regNode' is found, BDDs (local function, global function, BDD
  variable ID and auxillary BDD ID) at its fanin may no longer be
  necessary. nodeTable stores the pair <node,count>. 'count'
  indicates how may times the BDDs at that node will be accessed. Such
  BDDs are freed when appropriate, i.e., 'count' = 0.]

  SideEffects [None]

******************************************************************************/
void
SpfdReleaseAndCleanNodeData(
  SpfdApplData_t *applData,
  Ntk_Node_t *regNode,
  st_table *nodeTable)
{
  int k;
  Ntk_Node_t *fanin;
  st_table *inUseVars = applData->currInUseVars;
  st_table *regionNodes = applData->currRegionNodes;
  st_table *currBddReq = applData->currBddReq;
  bdd_manager *ddManager = applData->ddManager;
  char *dummy;

  Ntk_NodeForEachFanin(regNode,k,fanin) {
    int *count;
    if (st_lookup(nodeTable,(char *)fanin,&count)) {
      (*count)--;
      if (*count == 0) {
	int tempAuxId;
	int dummyInt;
	
	/* Free the auxId */
	tempAuxId = SpfdNodeReadAuxId(applData,fanin);
	if (st_delete(inUseVars,&tempAuxId,&dummy))
	  SpfdNodeSetAuxId(applData,fanin,-1);
	    
	if (st_lookup_int(regionNodes,(char *)fanin,&dummyInt)) {
	  /* If it is an internal node also delete the alternative function */
	  if (!dummyInt) 
	    SpfdNodeDeleteGlobalAlternative(applData,fanin);
	  /* If this is a region node delete the global Bdd */
	  if(st_delete(currBddReq,&fanin,&dummy))
	    bdd_recursive_deref(ddManager,(bdd_node *)dummy);
	} else {
	  if(st_delete(currBddReq,&fanin,&dummy))
	    bdd_recursive_deref(ddManager,(bdd_node *)dummy);
	}
	if (st_delete(nodeTable, &fanin, &count))
	   FREE(count);
      } /* End of *count == 0 */
    } /* st_lookup  */
  } /* Ntk_NodeForEachFanin */

  return;
  
} /* End of SpfdReleaseAndCleanNodeData */


/**Function********************************************************************

  Synopsis [Release those alternate PI ids that are no longer
  necessary. This is necessary as we resuse the BDD variable IDs.]

  SideEffects [None]

******************************************************************************/
void
SpfdReleaseAlternatePiIds(
  SpfdApplData_t *applData,
  Ntk_Node_t *regNode,
  st_table *nodeTable)
{
  int k;
  Ntk_Node_t *fanin;
  st_table *inUseVars = applData->currInUseVars;
  st_table *piAltVars = applData->currPiAltVars;
        
  Ntk_NodeForEachFanin(regNode,k,fanin) {
    int *count;
    if (st_lookup(nodeTable,(char *)fanin,&count)) {
      /* Do not delete the count. It is done in the final clean up
	 in SpfdReleaseAndCleanNodeDate. */
      if (*count == 1) {
	long tempId, tempAltId;
	tempId = Ntk_NodeReadMddId(fanin);
	if (st_lookup(piAltVars,(char *)(long)tempId,&tempAltId)) {
	  st_delete(inUseVars,&tempAltId,NIL(char *));
	  st_delete(piAltVars,&tempId,NIL(char *));
	}
      }
    }
  }

  return;
  
} /* End of SpfdReleaseAlternatePiIds */


/**Function********************************************************************

  Synopsis    [Return auxillay BDD ids to the pool.]

  SideEffects [None]

******************************************************************************/
  void
SpfdCleanUpAuxIds(SpfdApplData_t *applData)
{
  st_table *regionNodes = applData->currRegionNodes;
  st_generator *stGen;
  char *dummy;
  Ntk_Node_t *regNode,*fanin;
  int i;
  
  /* Release the auxIds */
  st_foreach_item(regionNodes,stGen,&regNode,&dummy) {
    Ntk_NodeForEachFanin(regNode,i,fanin) {
      SpfdNodeSetAuxId(applData,fanin,-1);
    }
  }

  /* Free the table storing the currently used BDD IDs for PI
     auxillary variables. */
  st_free_table(applData->currPiAltVars);
  applData->currPiAltVars = NIL(st_table);
  
  return;
} /* End of SpfdCleanUpAuxIds */


/*---------------------------------------------------------------------------*/
/* Definition of static functions                                            */
/*---------------------------------------------------------------------------*/

/**Function********************************************************************

  Synopsis           [Free SpfdNodeData_t]

  SideEffects        [None]

******************************************************************************/
static enum st_retval
NodeDataFree(
  char *key,
  char *value,
  char *arg)
{
  SpfdNodeData_t *data;
  bdd_manager *ddManager = (bdd_manager *)arg;

  data = (SpfdNodeData_t *)value;

  if (data) {
    if (data->spfd)
      bdd_recursive_deref(ddManager,data->spfd);
    if (data->alternative)
      bdd_recursive_deref(ddManager,data->alternative);
    if (data->faninOrder)
      array_free(data->faninOrder);
    if (data->parameters)
      FREE(data->parameters);
    if (data->localAlt)
      bdd_recursive_deref(ddManager,data->localAlt);
    FREE(data);
  }
    
  return (ST_CONTINUE);

} /* End of NodeDataFree */

