%{
/*
 *  vl2mv: Verilog to BLIF-MV Translator Distribution
 *
 *  Copyright (c) 1992, 1993
 *        Regents of the University of California
 *  All rights reserved.
 *
 *  Use and copying of this software and preparation of derivative works
 *  based upon this software are permitted.  However, any distribution of
 *  this software or derivative works must include the above copyright
 *  notice.
 *
 *  This software is made available AS IS, and neither the Electronics
 *  Research Laboratory or the Universify of California make any
 *  warranty about the software, its performance or its conformity to
 *  any specification.
 *
 *
 * $Header: /projects/development/hsv/CVSRepository/vl2mv/src/parser/verilog.l,v 1.4 2009/03/09 20:25:57 fabio Exp $
 *
 * Lexical Scanner for Verilog
 *
 * Author: Szu-Tsung Cheng, stcheng@ic.berkeley.edu
 *
 * Date: September 18, 1992
 */

#include <stdio.h>
#include "util.h"
#include "st.h"
#include "list.h"
#include "array.h"
#include "set.h"
#include "stack.h"
#include "vl_defs.h"
#include "vl_types.h"
#include "vlr_int.h"
#include "verilog_yacc.h"
#include "verilog.h"

#ifdef FLEX_SCANNER
#ifndef YY_FLEX_LEX_COMPAT
#define YYLMAX YY_BUF_SIZE
#else
#define MAX_INCLUDE_DEPTH 20
YY_BUFFER_STATE buffer_stack[MAX_INCLUDE_DEPTH];
int buffer_stack_ptr = 0;
#endif
#endif

extern vl_descPtr mod_list;
extern char *vl_filename;
extern st_table *aux_st;
extern int reserve_type_name;
extern int Loop_Unrolling;

static int scan_table = 0;
char yyid[MAXSTRLEN];
char brep[MAXSTRLEN];
int bexp0, bexp1;
char last_macro[MAXSTRLEN];  /* last macro name get expanded */
char curr_macro[MAXSTRLEN];
char yytypetoken[MAXSTRLEN];
st_table *macros;
char ifelse_stack[MAXSTRLEN];
int ifelseSP=0;

%}

%option yylineno

%a 3000
%e 1700
%k 1000
%n 700
%o 4000
%p 5000

%Start Snormal Stable Sskip

Space       [\f\n\r\t\b ]
Alpha       [a-zA-Z]
AlphaU      [a-zA-Z_]
AlphaNum    [a-zA-Z0-9]
AlphaNumU   [a-zA-Z0-9_]
Digit       [0-9]
DigitU      [0-9_]
Number      {Digit}{DigitU}*
Decimal     ({Number})?'[dD][ ]*{Number}
Octal       ({Number})?'[oO][ ]*[0-7xXzZ?][0-7xXzZ?_]*
Hexdecimal  ({Number})?'[hH][ ]*[0-9a-fA-FxXzZ?][0-9a-fA-FxXzZ?_]*
Binary      ({Number})?'[bB][ ]*[01xXzZ?][01xXzZ?_]*

%%

			     strcpy(last_macro, curr_macro);
			     curr_macro[0]='\0';

			     if (ifelseSP==0) {
			       if (scan_table)
				 BEGIN Stable;
			       else
				 BEGIN Snormal;
			     }

<Sskip,Snormal,Stable>{Space}+ {
#ifdef FLEX_SCANNER
				   int i;
				   for (i=0; i<yyleng; i++)
				       if (yytext[i] == '\n') yylineno++;
#endif
				   continue;
			       }

<Sskip,Snormal,Stable>`ifdef[^\n]*\n {
				 if (find_macro(yytext)) {
				     ifelse_stack[ifelseSP++] = 1;
				 } else {
				     ifelse_stack[ifelseSP++] = 0;
				     BEGIN Sskip;
				 }
				 if (ifelseSP == MAXSTRLEN)
				     lex_panic("nested `if/`else too deep");
			     }
<Sskip,Snormal,Stable>`ifndef[^\n]*\n {
				 if (find_not_macro(yytext)) {
				     ifelse_stack[ifelseSP++] = 1;
				 } else {
				     ifelse_stack[ifelseSP++] = 0;
				     BEGIN Sskip;
				 }
				 if (ifelseSP == MAXSTRLEN)
				     lex_panic("nested `if/`else too deep");
			     }
<Sskip,Snormal,Stable>`else  {
				 if (ifelse_stack[ifelseSP-1]) {
				     ifelse_stack[ifelseSP-1] = 0;
				     BEGIN Sskip;
				 } else {
				     if (scan_table)
					 BEGIN Stable;
				     else
					 BEGIN Snormal;
				 }
			     }
<Sskip,Snormal,Stable>`endif {
				 if (!ifelse_stack[--ifelseSP]) {
				     if (scan_table)
					 BEGIN Stable;
				     else
					 BEGIN Snormal;
				 }
			     }
<Sskip>[^\f\n\r\t\b ]+           { continue; }

<Snormal,Stable>"/*"         { skipcommentblock(); continue; }
<Snormal,Stable>"//"         { skipcommentline();  continue; }

<Snormal,Stable>`timescale[^\n]*\n { continue; }

<Snormal,Stable>`define[^\n]*\n {
#ifdef FLEX_SCANNER
				  yylineno++;
#endif
				  memorize_macro(yytext, macros);
				  continue;
				}
<Snormal,Stable>`include[^\n]*\n {   char *cp;
#ifdef FLEX_SCANNER
				     yylineno++;
#endif
				     yytext[strlen(yytext)-1] = '\0';
				     cp = strstr(yytext, "`include");
				     cp += strlen("`include");
				     while (*cp==' ' || *cp=='\t' || *cp=='\r')
					 cp++;
				     push_open_file(cp);
				 }

<Snormal,Stable>`VL2MVDataVar[^\n]*\n {
				    char *cp;
				    char varname[MAXSTRLEN],
					 vartype[MAXSTRLEN];
				    int i;
				    vl_id_range *id_sym;
#ifdef FLEX_SCANNER
				    yylineno++;
#endif
				    cp = strstr(yytext, "`VL2MVDataVar");
				    cp += strlen("`Vl2MVDataVar");
				    while (*cp==' '||*cp=='\t'||*cp=='\r')
					cp++;
				    i =0;
				    while (*cp!=' '&&*cp!='\t'&&*cp!='\r'&&
					   *cp!='\0') {
					varname[i++] = *cp;
					cp++;
				    }
				    varname[i] = '\0';
				    while (*cp==' '||*cp=='\t'||*cp=='\r')
					cp++;
				    i =0;
				    while (*cp!=' '&&*cp!='\t'&&*cp!='\r'&&
					   *cp!='\0'&&*cp!='\n') {
					vartype[i++] = *cp;
					cp++;
				    }
				    vartype[i] = '\0';
				    if (vl_currentModule) {
					if (st_lookup(vl_currentModule->sig_st,
						      varname,
						      (char**)&id_sym)) {
					    if (id_sym->unintType)
						vl_chk_free(id_sym->unintType);
					    id_sym->unintType =
						vlStrdup(vartype);
					}
				    }
				}

<Snormal,Stable>`VL2MVClock[^\n]*\n {  char *cp, cname[MAXSTRLEN],
					    buf[MAXSTRLEN];
				       int i;
				       int edge, pi;
				       float rho;
				       /* VL2MVClock 1:0:-1 0 0.0 */
#ifdef FLEX_SCANNER
				       yylineno++;
#endif
				       yytext[strlen(yytext)-1] = '\0';
				       cp = strstr(yytext, "`VL2MVClock");
				       cp += strlen("`VL2MVClock");
				       /* name */
				       while (*cp==' '||*cp=='\t'||*cp=='\r')
					   cp++;
				       i=0;
				       while (*cp!=' ' && *cp!='\t' &&
					      *cp!='\r' && *cp!='\0') {
					   cname[i++] = *cp; cp++;
				       }
				       cname[i] = '\0';

				       /* pi */
				       while (*cp==' '||*cp=='\t'||*cp=='\r')
					   cp++;
				       i=0;
				       while (*cp!=' ' && *cp!='\t' &&
					      *cp!='\r' && *cp!='\0') {
					   buf[i++] = *cp; cp++;
				       }
				       pi = atoi(buf);

				       /* edge */
				       while (*cp==' '||*cp=='\t'||*cp=='\r')
					   cp++;
				       i=0;
				       while (*cp!=' ' && *cp!='\t' &&
					      *cp!='\r' && *cp!='\0') {
					   buf[i++] = *cp; cp++;
				       }
				       edge = atoi(buf);

				       /* rho */
				       while (*cp==' '||*cp=='\t'||*cp=='\r')
					   cp++;
				       i=0;
				       while (*cp!=' ' && *cp!='\t' &&
					      *cp!='\r' && *cp!='\0') {
					   buf[i++] = *cp; cp++;
				       }
				       rho = atof(buf);

				       register_vl2mv_synClock(cname,
					   edge, pi, rho);
				       continue;
				    }

<Snormal,Stable>`VL2MVundefMod[^\n]*\n {   char *cp, modname[MAXSTRLEN];
					   int i;
#ifdef FLEX_SCANNER
					   yylineno++;
#endif
					   i=0;
					   yytext[strlen(yytext)-1] = '\0';
					   cp = strstr(yytext,
						       "`VL2MVundefMod");
					   cp += strlen("`VL2MVundefMod");
					   while (*cp==' ' || *cp=='\t' ||
						  *cp=='\r')
					       cp++;
					   while (*cp!=' ' && *cp!='\t' &&
						  *cp!='\r' && *cp!='\0') {
					       modname[i++] = *cp;
					       cp++;
					   }
					   modname[i] = '\0';
					   if (mod_list)
					       st_insert(mod_list->mod_holes,
							 vlStrdup(modname), 0);
					   continue;
				       }

<Snormal>">="               { return YYGEQ;  }
<Snormal>"=<"               { return YYLEQ;  }
<Snormal>"&&"               { return YYLOGAND;    }
<Snormal>"||"               { return YYLOGOR;     }
<Snormal>"==="              { return YYCASEEQUALITY;  }
<Snormal>"=="               { return YYLOGEQUALITY;   }
<Snormal>"!=="              { return YYCASEINEQUALITY; }
<Snormal>"!="               { return YYLOGINEQUALITY; }
<Snormal>"^~"               { return YYLOGXNOR; }
<Snormal>"~^"               { return YYLOGXNOR; }
<Snormal>"~&"               { return YYLOGNAND;      }
<Snormal>"~|"               { return YYLOGNOR;       }
<Snormal>"<<"               { return YYLSHIFT;      }
<Snormal>">>"               { return YYRSHIFT;      }
<Snormal>"?:"               { return YYCONDITIONAL; }

<Snormal>\"[^"]*\"          { return YYSTRING; }

<Snormal>always             { return YYALWAYS; }
<Snormal>"*>"               { return YYALLPATH; }
<Snormal>and                { return YYAND; }
<Snormal>assign             { return YYASSIGN; }
<Snormal>begin              { return YYBEGIN; }
<Snormal>buf                { return YYBUF; }
<Snormal>bufif0             { return YYBUFIF0; }
<Snormal>bufif1             { return YYBUFIF1; }
<Snormal>case               { return YYCASE; }
<Snormal>casex              { return YYCASEX; }
<Snormal>casez              { return YYCASEZ; }
<Snormal>cmos               { return YYCMOS; }
<Snormal>deassign           { return YYDEASSIGN; }
<Snormal>default            { return YYDEFAULT; }
<Snormal>defparam           { return YYDEFPARAM; }
<Snormal>disable            { return YYDISABLE; }
<Snormal>edge               { return YYEDGE; }
<Snormal>else               { return YYELSE; }
<Snormal>end                { return YYEND; }
<Snormal>endcase            { return YYENDCASE; }
<Snormal>endfunction        { return YYENDFUNCTION; }
<Snormal>endmodule          { return YYENDMODULE; }
<Snormal>endprimitive       { return YYENDPRIMITIVE; }
<Snormal>endspecify         { return YYENDSPECIFY; }
<Stable>endtable            { scan_table = 0; return YYENDTABLE; }
<Snormal>endtask            { return YYENDTASK; }
<Snormal>enum               { return YYENUM; }
<Snormal>event              { return YYEVENT; }
<Snormal>for                { return YYFOR; }
<Snormal>forever            { return YYFOREVER; }
<Snormal>fork               { return YYFORK; }
<Snormal>function           { return YYFUNCTION; }
<Snormal>highz0             { return YYHIGHZ0; }
<Snormal>highz1             { return YYHIGHZ1; }
<Snormal>if                 { return YYIF; }
<Snormal>initial            { return YYINITIAL; }
<Snormal>inout              { return YYINOUT; }
<Snormal>input              { return YYINPUT; }
<Snormal>integer            { return YYINTEGER; }
<Snormal>join               { return YYJOIN; }
<Snormal>large              { return YYLARGE; }
<Snormal>"=>"               { return YYLEADTO; }
<Snormal>macromodule        { return YYMACROMODULE; }
<Snormal>medium             { return YYMEDIUM; }
<Snormal>module             { return YYMODULE; }
<Snormal>mREG               { return YYMREG; }
<Snormal>"<="               { return YYNBASSIGN;  }
<Snormal>nand               { return YYNAND; }
<Snormal>negedge            { return YYNEGEDGE; }
<Snormal>nmos               { return YYNMOS; }
<Snormal>nor                { return YYNOR; }
<Snormal>not                { return YYNOT; }
<Snormal>notif0             { return YYNOTIF0; }
<Snormal>notif1             { return YYNOTIF1; }
<Snormal>or                 { return YYOR; }
<Snormal>output             { return YYOUTPUT; }
<Snormal>parameter          { return YYPARAMETER; }
<Snormal>pmos               { return YYPMOS; }
<Snormal>posedge            { return YYPOSEDGE; }
<Snormal>primitive          { return YYPRIMITIVE; }
<Snormal>pull0              { return YYPULL0; }
<Snormal>pull1              { return YYPULL1; }
<Snormal>pulldown           { return YYPULLDOWN; }
<Snormal>pullup             { return YYPULLUP; }
<Snormal>rcmos              { return YYRCMOS; }
<Snormal>real               { return YYREAL; }
<Snormal>reg                { return YYREG; }
<Snormal>repeat             { return YYREPEAT; }
<Snormal>"->"               { return YYRIGHTARROW; }
<Snormal>rnmos              { return YYRNMOS; }
<Snormal>rpmos              { return YYRPMOS; }
<Snormal>rtran              { return YYRTRAN; }
<Snormal>rtranif0           { return YYRTRANIF0; }
<Snormal>rtranif1           { return YYRTRANIF1; }
<Snormal>scalered           { return YYSCALARED; }
<Snormal>small              { return YYSMALL; }
<Snormal>specify            { return YYSPECIFY; }
<Snormal>specparam          { return YYSPECPARAM; }
<Snormal>strong0            { return YYSTRONG0; }
<Snormal>strong1            { return YYSTRONG1; }
<Snormal>supply0            { return YYSUPPLY0; }
<Snormal>supply1            { return YYSUPPLY1; }
<Snormal>swire              { return YYSWIRE; }
<Snormal>table              { scan_table = 1; return YYTABLE; }
<Snormal>task               { return YYTASK; }
<Snormal>teslaTimer         { return YYTESLATIMER; }
<Snormal>time               { return YYTIME; }
<Snormal>tran               { return YYTRAN; }
<Snormal>tranif0            { return YYTRANIF0; }
<Snormal>tranif1            { return YYTRANIF1; }
<Snormal>tri                { return YYTRI; }
<Snormal>tri0               { return YYTRI0; }
<Snormal>tri1               { return YYTRI1; }
<Snormal>triand             { return YYTRIAND; }
<Snormal>trior              { return YYTRIOR; }
<Snormal>trireg             { return YYTRIREG; }
<Snormal>typedef            { return YYTYPEDEF; }
<Snormal>vectored           { return YYVECTORED; }
<Snormal>wait               { return YYWAIT; }
<Snormal>wand               { return YYWAND; }
<Snormal>weak0              { return YYWEAK0; }
<Snormal>weak1              { return YYWEAK1; }
<Snormal>while              { return YYWHILE; }
<Snormal>wire               { return YYWIRE; }
<Snormal>wor                { return YYWOR; }
<Snormal>xnor               { return YYXNOR; }
<Snormal>xor                { return YYXOR; }

<Snormal>\$setup                { return YYsysSETUP; }
<Snormal>\$NDset                { return YYsysND; }
<Snormal>\$ND                   { return YYsysND; }
<Snormal>\${AlphaU}{AlphaNumU}* { return YYsysID; }

<Snormal>`{AlphaU}{AlphaNumU}* {
				char *macro;

				strcpy(yyid, yytext);
				/* macro expansion */
				if (st_lookup(macros, yyid+1, &macro)) {
				    expand_macro(yyid+1, macros);
				    continue;
				}
				return YYIDE;
			       }
<Snormal>{AlphaU}{AlphaNumU}* {
				vl_type *var_type;

				yytypetoken[0] = '\0';
				strcpy(yyid, yytext);

				/* type name checking */
				if (reserve_type_name)
				    if (st_lookup(mod_list->type_st,
						  yytext, (char**)&var_type))
					return YYuTYPE;

				/* enum name */
				if (aux_st) {
				    vl_enumerator *enum_elt;
				    if (st_lookup(aux_st,
						   yytext, (char**)&enum_elt)){
					sprintf(yytypetoken, yytext);
					sprintf(yytext, "%d", enum_elt->val);
					return YYINUMBER;
				    }
				}


				return YYIDE;
			      }
<Snormal>\\[^\n\t\b\r\f ]*    {
				strcpy(yyid, yytext);
				return YYIDE;
			      }

<Snormal>{Number}*\.{Number}+ { return YYRNUMBER; }
<Snormal>{Number}+\.{Number}* { return YYRNUMBER; }
<Snormal>{Number}             { return YYINUMBER; }
<Snormal>{Binary}             {
				binbin(yytext, brep);
				encodebit(brep, &bexp1, &bexp0);
				return YYINUMBER;
			      }
<Snormal>{Octal}              {
				octbin(yytext, brep);
				encodebit(brep, &bexp1, &bexp0);
				return YYINUMBER;
			      }
<Snormal>{Decimal}            {
				decbin(yytext, brep);
				encodebit(brep, &bexp1, &bexp0);
				return YYINUMBER;
			      }
<Snormal>{Hexdecimal}         {
				hexbin(yytext, brep);
				encodebit(brep, &bexp1, &bexp0);
				return YYINUMBER;
			      }

<Stable>\(\?\?\)              { return '*'; }
<Stable>\(01\)                { return 'r'; }
<Stable>\(10\)                { return 'f'; }
<Snormal,Stable>.             { return yytext[0]; }

%%

int yywrap()
{
    if (!empty_file_stack()) {
	pop_close_file();
	return 0;
    } else {
	return 1;
    }
}

void skipcommentblock()
{
    int done, level = 0;
    char c;

    for (done=0; !done; yyleng = (yyleng > YYLMAX-2) ? YYLMAX-2 : yyleng) {
	if ((c = input()) == '*') {
	    yytext[yyleng++] = c;
	    if ((c = input()) == '/') {
		done = (level-- == 0);
	    } else {
		unput(c);
	    }
	} else if (c == '/') {
	    yytext[yyleng++] = c;
	    if ((c = input()) == '*') {
		level++;
	    } else {
		unput(c);
	    }
	} else if (c == 0) {
	    char buf[MAXSTRLEN];
	    sprintf(buf, "incomplete comment (%d)\n", yylineno);
	    fail(buf);
	} else {
#ifdef FLEX_SCANNER
	    if (c == '\n') yylineno++;
#endif
	    yytext[yyleng++] = c;
	}
    }
}

void skipcommentline()
{
    int done;
    char c;

    for (done=0; !done; yyleng = (yyleng > YYLMAX-2) ? YYLMAX-2 : yyleng) {
	if ((c = input()) == '\n') {
	    done = 1;
	} else {
	    yytext[yyleng++] = c;
	}
    }
    yytext[yyleng] = '\0';
#ifdef FLEX_SCANNER
    yylineno++;
#endif

}

void binbin(instr, outstr)
char *instr;
char *outstr;
{
    char *cp, *firstcp;
    int blen = 0, bpos=0;

    blen = atoi(instr);
    blen = (blen==0) ? MAXBITNUM : blen;
    firstcp = strpbrk(instr, "bB")+1;
    cp = instr + strlen(instr) - 1;
    outstr[blen] = '\0';
    for (bpos = blen-1; bpos >=0 && cp >= firstcp; cp--) {
	if (*cp != '_' && *cp != ' ') {
	    outstr[bpos] = *cp;
	    bpos--;
	}
    }
    for (; bpos >=0; bpos--) {
	outstr[bpos] = '0';
    }
}

char *hexnum[16] = {"0000","0001","0010","0011",
		    "0100","0101","0110","0111",
		    "1000","1001","1010","1011",
		    "1100","1101","1110","1111"};

void decbin(instr, outstr)
char *instr;
char *outstr;
{
    char *firstcp, buf[MAXSTRLEN];
    int num, blen = 0;

    utol(instr);
    blen = atoi(instr);
    blen = (blen==0) ? MAXBITNUM : blen;
    firstcp = strpbrk(instr, "dD")+1;
    while (*firstcp==' ') firstcp++;
    num = atoi(firstcp); /* don't put x, z, ? in decimal string */
    sprintf(buf, "%d'h%x", blen, num);
    hexbin(buf, outstr);
}

void octbin(instr, outstr)
char *instr;
char *outstr;
{
    char *cp, *firstcp;
    int blen = 0, bpos=0, i;

    utol(instr);
    blen = atoi(instr);
    blen = (blen==0) ? MAXBITNUM : blen;
    firstcp = strpbrk(instr, "oO")+1;
    cp = instr + strlen(instr) - 1;
    outstr[blen] = '\0';
    for (bpos = blen-1; bpos >=0 && cp >= firstcp; cp--) {
	if (*cp != '_' && *cp != ' ') {
	    for (i = 3; i >= 1; i--) {
		if (bpos >= 0) {
		    if (*cp=='x' || *cp=='z' || *cp=='?') {
			outstr[bpos] = *cp;
			bpos--;
		    } else if (isdigit(*cp)) {
			outstr[bpos] = hexnum[*cp-'0'][i];
			bpos--;
		    }
		}
	    }
	}
    }
    for (; bpos >=0; bpos--) {
	outstr[bpos] = '0';
    }
}

void hexbin(instr, outstr)
char *instr;
char *outstr;
{
    char *cp, *firstcp;
    int blen = 0, bpos=0, i;

    utol(instr);
    blen = atoi(instr);
    blen = (blen==0) ? MAXBITNUM : blen;
    firstcp = strpbrk(instr, "hH")+1;
    cp = instr + strlen(instr) - 1;
    outstr[blen] = '\0';
    for (bpos = blen-1; bpos >=0 && cp >= firstcp; cp--) {
	if (*cp != '_' && *cp != ' ') {
	    for (i = 3; i >= 0; i--) {
		if (bpos >= 0) {
		    if (*cp=='x' || *cp=='z' || *cp=='?') {
			outstr[bpos] = *cp;
			bpos--;
		    } else if (isdigit(*cp)) {
			outstr[bpos] = hexnum[*cp-'0'][i];
			bpos--;
		    } else if (isxdigit(*cp)) {
			outstr[bpos] = hexnum[*cp-'a'+10][i];
			bpos--;
		    }
		}
	    }
	}
    }
    for (; bpos >=0; bpos--) {
	outstr[bpos] = '0';
    }
}

char *utol(str)
char *str;
{
    char *cp;

    for (cp = str; *cp!='\0'; cp++) {
	if (isupper(*cp)) *cp = tolower(*cp);
    }
    return str;
}

void encodebit(char *instr, int *part1, int *part0)
{
    int i, bmask;

    *part1 = *part0 = 0;
    i = strlen(instr);
    i = (i > MAXBITNUM) ? MAXBITNUM-1 : i-1;
    for (bmask = 1; i>=0; bmask <<= 1, i--) {
	switch (instr[i]) {
	case '1': *part1 |= bmask; break;
	case '0': *part0 |= bmask; break;
	case 'x':
	case '?': *part1 |= bmask; *part0 |= bmask; break;
	case 'z': break;
	}
    }
}

/*
 * memorize_macro
 *
 *     insert a argumentless macro into macro table
 */
void memorize_macro(instr, macros)
char *instr;
st_table *macros;
{
    char *cp, *cp1, *cp2;
    char mname[MAXSTRLEN];

    cp = strstr(instr,MACRO_DEFINE) + strlen(MACRO_DEFINE);
    sscanf(cp, "%s", mname);
    cp = strstr(cp, mname) + strlen(mname);
    cp1 = strstr(cp, "//");
    cp2 = strstr(cp, "/*");
    if (cp1 || cp2) {
	if (cp1) *cp1='\0';
	if (cp2) *cp2='\0';
    }

    st_insert(macros, vlStrdup(mname), vlStrdup(cp));
}

/*
 * expand_macro
 *
 *     expand an argumentless macro into input stream (using unput)
 */
void expand_macro(mname, macros)
char *mname;
st_table *macros;
{
    char *mbody;
    int i;

    if (st_lookup(macros, mname, &mbody)) {
	for (i=strlen(mbody)-1; i>=0; i--) {
	    unput(mbody[i]);
	}
	strcpy(curr_macro, mname);
    }
}

stack_t file_stack;
stack_t fname_stack;
stack_t lineno_stack;
stack_t buffer_stack;

void push_open_file(char *fname)
{
    FILE *infile;
    char *filename;
    char buf[MAXSTRLEN];

    filename = vlStrdup(fname);
    if (*filename == '"') {
	unsigned int i, j;
	j = strlen(filename)-1;
	while (filename[j] == ' ' || filename[j] == '\t' ||
	       filename[j] == '\r') j--;
	if (filename[j] != '"') {
	    sprintf(buf, "unbalanced \"\n");
	    yyerror(buf);
	}
	filename[j] = '\0';
	for(i=0; i<strlen(filename); i++) filename[i] = filename[i+1];
    }
    if ((infile = fopen(filename,"r"))==NULL) {
	sprintf(buf, "fail to open file '%s'", filename);
	yyerror(buf);
    }

    pushd(filename);

    push_stack(file_stack, (void*)yyin);
    push_stack(fname_stack, (void*)vl_filename);
    push_stack(lineno_stack, (void*)yylineno);

#if defined(FLEX_SCANNER) && !defined(YY_FLEX_LEX_COMPAT)
    push_stack(buffer_stack, (void*) YY_CURRENT_BUFFER);
    yy_switch_to_buffer(yy_create_buffer(infile, YY_BUF_SIZE));
#else
    yyin = infile;
#endif
    yylineno = 1;
    vl_filename = filename;
}

void pop_close_file()
{
    FILE *infile;
    int old_yylineno;
    char *filename;
#ifdef FLEX_SCANNER
    YY_BUFFER_STATE buffer;
#endif

    fclose(yyin);
    pop_stack(file_stack, (void**)&infile);
    pop_stack(fname_stack, (void**)&filename);
    pop_stack(lineno_stack, (void**)&old_yylineno);
    popd();
#if defined(FLEX_SCANNER) && !defined(YY_FLEX_LEX_COMPAT)
    yy_delete_buffer(YY_CURRENT_BUFFER);
    pop_stack(buffer_stack, (void**)&buffer);
    yy_switch_to_buffer(buffer);
#else
    yyin = infile;
#endif
    yylineno = old_yylineno;
    vl_chk_free(vl_filename);
    vl_filename = filename;
}

int empty_file_stack()
{
    return (lsLength(file_stack)==0);
}

int find_macro(char *str)
{
    char *cp;
    char *dummy;
    char mname[MAXSTRLEN];

    cp = strstr(str,MACRO_IFDEF) + strlen(MACRO_IFDEF);
    sscanf(cp, "%s", mname);
    return (st_lookup(macros, mname, &dummy));
}

int find_not_macro(char *str)
{
    char *cp;
    char *dummy;
    char mname[MAXSTRLEN];

    cp = strstr(str,MACRO_IFNDEF) + strlen(MACRO_IFNDEF);
    sscanf(cp, "%s", mname);
    return (!st_lookup(macros, mname, &dummy));
}

void lex_panic(char *mesg)
{
    fprintf(stderr, "%s\n", mesg);
    exit(1);
}


static stack_t dir_stack=0;

void pushd(char *filename)
{
    int i;
    char cwd[MAXSTRLEN], pwd[MAXSTRLEN];

    if (!dir_stack) dir_stack = create_stack();

    getcwd(pwd, MAXSTRLEN-1);

    for (i=strlen(filename)-1; i>=0; i--)
	if (filename[i] == '/') break;

    strcpy(cwd, filename);
    if (i==-1) { cwd[0]='.'; cwd[1]='\0'; }
    else      { cwd[i]='\0'; }

    push_stack(dir_stack, vlStrdup(pwd));
    chdir(cwd);
}

void popd()
{
    char *cwd;

    pop_stack(dir_stack, (void**)&cwd);
    chdir(cwd);
    free(cwd);
}
